/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * args.c - Process command-line arguments
 */

/*
 * If you add a new option here and its value can change while spettro
 * is running, you probably want to adjust main() so that its value is
 * preserved across the exec() that spettro does when skilling to next file.
 */

#include "spettro.h"
#include "args.h"

#include "a_file.h"
#include "barlines.h"
#include "colormap.h"
#include "convert.h"
#include "ui.h"
#include "usage.h"
#include "version.h"

/* Stuff for getopt() */
#include <unistd.h>
#include <getopt.h>
extern char *optarg;
extern int optind, opterr, optopt;

#include <ctype.h>	/* for tolower() */
#include <errno.h>

#if ECORE_MAIN
#include <Ecore.h>	/* For EFL_VERSION_* */
#endif

/* Hack to handle window and colormap long-form arguments */
#ifndef CTRL
# define CTRL(c) ((c) & 31)
#endif
#ifndef META
# define META(c) ((c) | 128)
#endif

/* Print a summary of the command-line flags */
static void
usage(void)
{
    printf(
"Usage: spettro [options] file\n\
n=number, t=time, c=character, s=string, f=frequency or note name\n\
times can be in seconds, mins:secs or hours:mins:secs plus .fraction-of-second\n\
-p    Pause on startup and at the end of the audio file, where Space replays\n\
-S    Play the files in a random order\n\
-m    Be mute\n\
-w n  Set the window's width to n pixels, default %u\n\
-h n  Set the window's height to n pixels, default %u\n",
			disp_width, disp_height); printf("\
-F    Start up in fullscreen mode\n\
-Z    Use SDL2's software renderer: slower but works over ssh\n\
-U    Start up minimized\n\
-e    Start up showing the whole audio file\n\
-n f  Set the minimum displayed frequency, default %g\n\
-x f  Set the maximum displayed frequency, default %g\n",
				DEFAULT_MIN_FREQ, DEFAULT_MAX_FREQ); printf("\
-z n  Set the dynamic range of the the color map in decibels, default %g\n",
				DEFAULT_DYN_RANGE); printf("\
-i n  Set the intensity of the brightest color in dB, default %g\n",
				DEFAULT_LOGMAX); printf("\
-a    Show the frequency axes\n\
-A    Show the time axis and the status line\n\
-T    Use the tiny 3x5 font for the axes\n\
-f n  Set the FFT frequency in Hz, default %g, minimum %g\n",
				DEFAULT_FFT_FREQ, MIN_FFT_FREQ); printf("\
-j n  Use n FFT calculation threads, default: the number of CPUs\n\
-t t  Set the initial playing time\n\
-l t  Set the time for the left bar line\n\
-r t  Set the time for the right bar line\n\
-b n  Set the number of beats per bar\n\
-P n  Set how many pixel columns to display per second of audio, default %g\n",
				DEFAULT_PPSEC); printf("\
-R n  Set the scrolling rate in frames per second, default %g\n",
				DEFAULT_FPS); printf("\
-k    Overlay black and white lines showing frequencies of an 88-note keyboard\n\
-s    Overlay conventional score notation pentagrams\n\
-g    Overlay lines showing the frequencies of a classical guitar's strings\n\
-v n  Set the soft volume level to n (>1.0 is louder, <1.0 is softer)\n\
-D t  Gradually increase softvol so that it Doubles every t seconds\n\
-W c  Use FFT window function x where x starts with\n\
      K for Kaiser, D for Dolph, N for Nuttall, B for Blackman, H for Hann\n\
-c c  Select a color map: Heatmap, Gray or Print\n\
-o s  Display the spectrogram, dump it to file s in PNG format and quit\n\
-V    Which version of spettro is this, and which libraries does it use?\n\
-K    Show which key presses do what\n\
-H    This!\n\
");
}

static void
show_keys()
{
    char **cpp; 	/* Pointer to string */

    printf("== Keyboard commands ==\n");
    for (cpp=usage_keys; *cpp != NULL; cpp++)
	printf("%s\n", *cpp);

    printf("== Mouse controls ==\n\
Left/Right click (and drag): Set the position of the left/right bar line\n\
");
}

/*
 * Process command-line options, returning the index into argv of the first
 * filename or -1 if there was a syntax error. If no filenames were supplied,
 * argv[return-value] will be NULL.
 */
void
process_args(int argc, char **argv)
{
    /* Local versions to delay setting until audio length is known */
    secs_t bar_left_time = UNDEFINED;
    secs_t bar_right_time = UNDEFINED;

    int opt;
#if HAVE_GETOPT_LONG
    int longindex = 0;
    const struct option longopts[] = {
	{ "width",	1, NULL, 'w' },
	{ "height",	1, NULL, 'h' },
	{ "jobs",	1, NULL, 'j' },
	{ "left",	1, NULL, 'l' },
	{ "right",	1, NULL, 'r' },
	{ "beats",	1, NULL, 'b' },
	{ "fft-freq",	1, NULL, 'f' },
	{ "start",	1, NULL, 't' },
	{ "output",	1, NULL, 'o' },
	{ "window",	1, NULL, 'W' },
	{ "colormap",	1, NULL, 'c' },
	{ "colourmap",	1, NULL, 'c' },
	/* long forms of "-WK", "-WD" etc */
	{ "kaiser",	0, NULL, CTRL('K') },
	{ "dolph",	0, NULL, CTRL('D') },
	{ "nuttall",	0, NULL, CTRL('N') },
	{ "blackman",	0, NULL, CTRL('B') },
	{ "hann",	0, NULL, CTRL('H') },
	/* long forms of "-ch", "-cg", "-cp" */
	{ "heat",	0, NULL, META('H') },
	{ "gray",	0, NULL, META('G') },
	{ "grey",	0, NULL, META('G') },
	{ "print",	0, NULL, META('P') },
	{ "softvol",	1, NULL, 'v' },
	{ "double",	1, NULL, 'D' },
	{ "dyn-range",	1, NULL, 'z' },
	{ "maxdb",	1, NULL, 'i' },
	{ "min-freq",	1, NULL, 'n' },
	{ "max-freq",	1, NULL, 'x' },
	/* Boolean options */
	{ "pause",	0, NULL, 'p' },
	{ "shuffle",	0, NULL, 'S' },
	{ "mute",	0, NULL, 'm' },
	{ "fullscreen",	0, NULL, 'F' },
	{ "software-renderer",	0, NULL, 'Z' },
	{ "minimized",	0, NULL, 'U' },
	{ "minimised",	0, NULL, 'U' },
	{ "fit",	0, NULL, 'e' },
	{ "expand",	0, NULL, 'e' },
	{ "piano",	0, NULL, 'k' },
	{ "pianokeys",	0, NULL, 'k' },
	{ "guitar",	0, NULL, 'g' },
	{ "score",	0, NULL, 's' },
	{ "frequency-axis", 0, NULL, 'a' },
	{ "time-axis",	0, NULL, 'A' },
	{ "tiny-font",	0, NULL, 'T' },
	/* long-form-only options, with arbitrary one-char codes */
	{ "version",	0, NULL, 'V' },
	{ "help",	0, NULL, 'H' },
	{ "keys",	0, NULL, 'K' },
	{ NULL,		0, NULL,  0  },
    };

    while ((opt = getopt_long(argc, argv,
#elif HAVE_GETOPT
    while ((opt = getopt(argc, argv,
#else
# error "Neither getopt() nor getopt_long() was found"
#endif
	"pSmFZUeksgaATVHKw:h:j:b:n:x:f:v:D:z:R:P:i:t:l:r:o:W:c:"
#if HAVE_GETOPT_LONG
	, longopts, &longindex
#endif
	   )) != -1)
    switch (opt) {
    /*
     * getopt() error codes.
     */
    /* getopt_long() doesn't set longindex if a long option is missing its
     * obligatory parameter, so we can't tell if they said --height or -h.
     * Only "optopt" is set, so we can only report the character argument.
     * Of these alternatives, having getopt() print its error message:
     * ./spettro: option requires an argument -- 'P'
     * seems the least bad.
     */
    case '?':	/* Unrecognised option */
	fprintf(stdout, "spettro -H gives a list of valid options.\n");
	exit(1);
    /*
     * Help options (a.k.a. --help, --keys and --version)
     */
    case 'H':
	usage();
	exit(0);
    case 'K':
	show_keys();
	exit(0);
    case 'V':
	print_version();
	exit(0);
    /*
     * Boolean flags
     */
    case 'p':
	autoplay_mode = FALSE;
	break;
    case 'S':
	shuffle_mode = TRUE;
	break;
    case 'm':
	mute_mode = TRUE;
	break;
    case 'F':
	fullscreen_mode = TRUE;
	break;
    case 'Z':
	software_renderer = TRUE;
	break;
    case 'U':
	start_up_minimized = TRUE;
	break;
    case 'e':
	start_up_fit = TRUE;
	break;
    case 'k':	/* Draw black and white lines where piano keys fall */
	piano_lines = TRUE;
	break;
    case 's':	/* Draw conventional score notation staff lines */
	staff_lines = TRUE;
	guitar_lines = FALSE;
	break;
    case 'g':	/* Draw guitar string lines */
	guitar_lines = TRUE;
	staff_lines = FALSE;
	break;
    case 'a':
	show_frequency_axes = TRUE;
	break;
    case 'A':
	show_time_axes = TRUE;
	break;
    case 'T':
	use_tiny_font = TRUE;
	break;
    /*
     * Parameters that take an integer argument
     */
    case 'w':
	if ((disp_width = atoi(optarg)) <= 0) {
	    fprintf(stdout, "-w width must be > 0\n");
	    exit(1);
	}
	break;
    case 'h':
	if ((disp_height = atoi(optarg)) <= 0) {
	    fprintf(stdout, "-h height must be > 0\n");
	    exit(1);
	}
	break;
    case 'j':
	if ((max_threads = atoi(optarg)) < 0) {
	    fprintf(stdout, "-j threads must be >= 0\n");
	    exit(1);
	}
	break;
    case 'b':
	if ((beats_per_bar = atoi(optarg)) < 0) {
	    fprintf(stdout, "-b beats_per_bar must be >= 0\n");
	    exit(1);
	}
	/* Let them say 0 to mean "no beats" (which should be 1) */
	if (beats_per_bar == 0) beats_per_bar = DEFAULT_BEATS_PER_BAR;
	break;

    /*
     * Parameters that take a floating point argument
     */
    case 'n':	/* Minimum frequency */
    case 'x':	/* Maximum frequency */
    case 'f':	/* Set FFT frequency */
    case 'v':	/* Set software volume control */
    case 'z':	/* Set dynamic range */
    case 'R':	/* Set scrolling rate */
    case 'P':	/* Set pixel columns per second */
    case 'i':	/* Set logmax */
	errno = 0;
	{
	    double arg;

	    /* Min and max frequencies can also be "A0" or whatever */
	    if ((opt == 'n' || opt == 'x') &&
		(arg = note_name_to_freq(optarg)) != 0.0) {
		;
	    } else {
		float f;

		if (sscanf(optarg, "%f", &f) == 1) {
		    arg = f;
		} else {
		    fprintf(stdout, "The parameter to -%c must be a ",
				    opt);
		    switch (opt) {
		    case 'n':	/* Minimum frequency */
		    case 'x':	/* Maximum frequency */
			fprintf(stdout, "frequency in Hz or a note name");
			break;
		    case 'f':	/* Set FFT frequency */
			fprintf(stdout, "frequency in Hz");
			break;
		    case 'i':	/* Set logmax */
			fprintf(stdout, "value in dB");
			break;
		    case 'z':	/* Set dynamic range */
			fprintf(stdout, "range in dB");
			break;
		    case 'v':	/* Set software volume control */
		    case 'P':	/* Set pixel columns per second */
		    case 'R':	/* Set scrolling rate */
			fprintf(stdout, "floating point number");
			break;
		    }
		    fprintf(stdout, ".\n");
		    exit(1);
		}
	    }
	    /* They should all be >= 0 except for logmax */
	    if (arg < 0.0 && opt != 'i') {
		fprintf(stdout, "The argument to -%c must be positive.\n",
			opt);
		exit(1);
	    }

	    /* Place an arbitrary lower limit on the FFT frequency */
	    if (opt == 'f' && DELTA_LT(arg, MIN_FFT_FREQ)) {
		fprintf(stdout, "The FFT frequency must be >= %g, not %g (optarg=\"%s\")\n",
			MIN_FFT_FREQ, arg, optarg);
		exit(1);
	    }

	    /* These must be > 0.
	     * Dynamic range and FPS can be 0, if silly.
	     */
	    if (arg == 0.0) switch (opt) {
	    case 'f': case 'n': case 'x': case 'P':
		fprintf(stdout, "The argument to -%c must be positive.\n",
			opt);
		exit(1);
	    default:
		break;
	    }
	    switch (opt) {
	    case 'n': min_freq = arg;	break;
	    case 'x': max_freq = arg;	break;
	    case 'f': fft_freq = arg;	break;
	    case 'v': softvol = arg;	break;
	    case 'z': dyn_range = arg;	break;
	    case 'R': fps = arg;		break;
	    case 'P': ppsec = arg;		break;
	    case 'i': logmax = arg;		break;
	    default: fprintf(stdout, "Internal error: Unknown numeric argument -%c\n", opt);
	    }
	}
	break;
    /*
     * Parameters that take a time argument
     */
    case 't':	/* Play starting from time t */
    case 'l':	/* Set left bar line position */
    case 'r':	/* Set right bar line position */
    case 'D':	/* Double softvol in n seconds */
	{
	    secs_t secs = string_to_seconds(optarg);

	    if (secs < 0.0) {
		fprintf(stdout, "Time not recognized in -%c %s; ",
		    opt, optarg);
		fprintf(stdout, "the maximum is 99:59:59.99 (359999.99 seconds).\n");
		exit(1);
	    }

	    switch (opt) {
		case 't': start_time = secs;	 break;
		case 'l': bar_left_time = secs;	 break;
		case 'r': bar_right_time = secs; break;
		case 'D': softvol_double = secs; break;
	    }
	}
	break;

    /*
     * Parameters that take a string argument
     */
    case 'o':
	output_file = optarg;
	break;

    case 'W':
	switch (tolower(optarg[0])) {
	case 'k': window_function = KAISER; break;
	case 'n': window_function = NUTTALL; break;
	case 'h': window_function = HANN; break;
	case 'b': window_function = BLACKMAN; break;
	case 'd': window_function = DOLPH; break;
	default:
	    fprintf(stdout, "-W which? Kaiser, Dolph, Nuttall, Blackman or Hann?\n");
	    exit(1);
	}
	break;
    /* long form equivalents */
    case CTRL('K'): window_function = KAISER; break;
    case CTRL('D'): window_function = DOLPH; break;
    case CTRL('N'): window_function = NUTTALL; break;
    case CTRL('B'): window_function = BLACKMAN; break;
    case CTRL('H'): window_function = HANN; break;

    case 'c':			     /* Choose color map */
	switch (tolower(optarg[0])) {
	case 'h': set_colormap(HEAT_MAP); break;
	case 'g': set_colormap(GRAY_MAP); break;
	case 'p': set_colormap(PRINT_MAP); break;
	default:
	    fprintf(stdout, "-m: Which colormap? (heat/gray/print)\n");
	    exit(1);
	}
	break;
    /* long form equivalents */
    case META('H'): set_colormap(HEAT_MAP); break;
    case META('G'): set_colormap(GRAY_MAP); break;
    case META('P'): set_colormap(PRINT_MAP); break;
    default:
	fprintf(stdout, "getopt returned an unknown option %d\n", opt);
	break;
    }

    /* Don't call set_*_bar_time because that would trigger repaints before
     * the graphics system is up */
    if (bar_left_time != UNDEFINED) {
	left_bar_time = bar_left_time;
    }
    if (bar_right_time != UNDEFINED) {
	right_bar_time = bar_right_time;
    }

    /* Sanity checks */

    /* Upside-down graphs (tho' it works!) and ranges that are too tiny */
    if (max_freq - min_freq < 1.0) {
	fprintf(stdout, "The maximum frequency must be higher than the minimum!\n");
	exit(1);
    }
}

/* Parse environment variable SPETTROFLAGS as if it were arg[cv] */
void
process_spettroflags(void)
{
    char *av[256];	/* Fake argv */
    int ac;		/* Fake argc */
    char *spettroflags; /* The environment's version */
    char *cp;

    spettroflags = getenv("SPETTROFLAGS");
    if (!spettroflags) return;

    /* Make a local copy to modify */
    spettroflags = strdup(spettroflags);
    if (!spettroflags) return;

    /* Fake argv[0] */
    av[0] = "spettro"; ac = 1;

    cp = spettroflags;
    while (*cp != '\0') {
	av[ac++] = cp;
	cp = strchr(cp, ' ');
	if (cp == NULL) break;
	*cp++ = '\0';
	/* Skip multiple spaces */
	while (*cp == ' ') cp++;
    }
    av[ac] = NULL;
    process_args(ac, av);
}
