/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * audio.c - Audio-playing functions
 */

#include "spettro.h"
#include "audio.h"

#include "a_cache.h"
#include "axes.h"
#include "convert.h"		/* for frames_to_string() */
#include "gui.h"
#include "lock.h"
#include "ui.h"

#include <sys/time.h>	/* for gettimeofsay() */


#if EMOTION_AUDIO

#include <Emotion.h>
extern Evas_Object *em;
static void playback_finished_cb(void *data, Evas_Object *obj, void *ev);

#elif SDL_AUDIO

#include <SDL.h>
static frames_t sdl_start = 0;	/* At what offset in the audio file, in frames,
				 * will we next read samples to play? */
static void sdl_fill_audio(void *userdata, Uint8 *stream, int len);
static frames_t sdl_buffer_size;	/* In sample frames */

#endif

static void remember_real_playing_time(secs_t when);

enum playing playing = PAUSED;

void
audio_init(audio_file_t *af, char *filename)
{
#if EMOTION_AUDIO
    /* Set audio player callbacks */
    evas_object_smart_callback_add(em, "playback_finished",
				   playback_finished_cb, NULL);

    /* Load the audio file for playing */
    emotion_object_init(em, NULL);
    emotion_object_video_mute_set(em, EINA_TRUE);
    if (emotion_object_file_set(em, filename) != EINA_TRUE) {
	fputs("Enlightenment couldn't load the audio file.\n", stdout);
	exit(1);
    }
    evas_object_show(em);
#elif SDL_AUDIO
    {
	freq_t sample_rate = af->sample_rate;
	SDL_AudioSpec wavspec;

	wavspec.freq = round(sample_rate);
	wavspec.format = AUDIO_S16SYS;
	wavspec.channels = af->channels;
	/* 4096 makes for a visible lag between audio and video, as the video
	 * follows the next af-reading position, which is 0-4096 samples
	 * ahead of what's playing now.
	 * Set it to "secpp" so that we should never get more than one column
	 * behind.
	 */
	if (sample_rate == 0.0) {
	    fprintf(stdout, "Internal error: audio_init() was called before \"sample_rate\" was initialized.\n");
	    exit(1);
	}
	/* The SDL buffer size determines the precision with which we return
	 * the current playing time, hence influences the time-graininess of
	 * the scrolling. Setting it to fps (default 1/25s) makes the scrolling
	 * jiggle, presumably as SDL-frame and pixel-column boundaries coincide
	 * and not and I doubt we can change the audio buffer size while it's
	 * playing, so we fix it at something much smaller, a quarter of that.
	 */
	wavspec.samples = sample_rate * af->channels / 100;
	/* SDL sometimes requires a power-of-two buffer,
	 * failing to work if it isn't, so reduce it to such. */
	{
	    int places = 0;
	    while (wavspec.samples > 0) {
		wavspec.samples >>= 1;
		places++;
	    }
	    wavspec.samples = 1 << places;
	}
	sdl_buffer_size = wavspec.samples / af->channels;  /* Size in frames */
	wavspec.callback = sdl_fill_audio;
	wavspec.userdata = af;

	if (SDL_InitSubSystem(SDL_INIT_AUDIO) != 0) {
	    fprintf(stdout, "Couldn't initialize SDL audio: %s\n",
		    SDL_GetError());
	    exit(1);
	}

	if (SDL_OpenAudio(&wavspec, NULL) < 0) {
	    fprintf(stdout, "Couldn't open SDL audio: %s.\n", SDL_GetError());
	    exit(1);
	}
    }
#else
# error "Define one of EMOTION_AUDIO and SDL_AUDIO"
#endif
}

/* Turn the audio device driving system off */
void
audio_deinit()
{
    /* Stop playing */
    if (playing == PLAYING) {
	pause_audio();
    }
#if EMOTION_AUDIO
    /* Emotion doesn't seem to have a way to turn the audio off */
#elif SDL_AUDIO
    SDL_QuitSubSystem(SDL_INIT_AUDIO);
#endif
}

#if EMOTION_AUDIO
/*
 * Callback is called when the player gets to the end of the piece.
 *
 * The "playback_started" event is useless because in emotion 0.28 it is
 * delivered when playback of audio finishes (!)
 * An alternative would be the "decode_stop" callback but "playback_finished"
 * is delivered first.
 */
static void
playback_finished_cb(void *data, Evas_Object *obj, void *ev)
{
    stop_playing();
}
#endif

void
pause_audio()
{
    if (!mute_mode)
#if EMOTION_AUDIO
	emotion_object_play_set(em, EINA_FALSE);
#elif SDL_AUDIO
	SDL_PauseAudio(1);
#endif
    remember_real_playing_time(get_playing_time());
    playing = PAUSED;
}

/* Start playing the audio again from it's current position */
void
start_playing()
{
#if EMOTION_AUDIO
    if (!mute_mode) emotion_object_play_set(em, EINA_TRUE);
#endif
#if SDL_AUDIO
    if (!mute_mode) SDL_PauseAudio(0);
#endif
    set_real_start_time(disp_time);
    playing = PLAYING;
}

/* Stop playing because it has arrived at the end of the piece */
void
stop_playing()
{
#if EMOTION_AUDIO
    if (!mute_mode) emotion_object_play_set(em, EINA_FALSE);
#endif
#if SDL_AUDIO
    if (!mute_mode) SDL_PauseAudio(1);
#endif

    playing = STOPPED;

    /* Avoid it sometimes overlaying playing_time and end-of-file time
     * with values differing by 1/100th of a second.
     */
    if (show_time_axes) draw_time_axis();

    if (autoplay_mode) {
	nexting = TRUE;
	gui_quit_main_loop();
    }
}

void
continue_playing()
{
    if (!mute_mode) {
#if EMOTION_AUDIO
	/* Resynchronise the playing position to the display,
	 * as emotion stops playing immediately but seems to throw away
	 * the unplayed part of the currently-playing audio buffer.
	 */
	emotion_object_position_set(em, disp_time);
	emotion_object_play_set(em, EINA_TRUE);
#endif
#if SDL_AUDIO
	sdl_start = round(disp_time * sr);
	SDL_PauseAudio(0);
#endif
    }
    set_real_start_time(disp_time);
    playing = PLAYING;
}

/*
 * SDL and Emotion's reporting of the current playing time is grainy,
 * making the scrolling jittery. Instead we calculate it using the
 * real time clock, hoping that it remains in sync.
 */
static secs_t real_start_time;		 /* When we started playing from 0.0,
					  * in seconds from the epoch */
static secs_t real_playing_time;	 /* When we pause in mute mode, remember
					  * where we are in the piece */
static bool use_real_start_time = FALSE; /* Should we use real_start_time? */

/*
 * Position the audio player at the specified time in seconds.
 */
void
set_playing_time(secs_t when)
{
    if (!mute_mode) {
#if EMOTION_AUDIO
	emotion_object_position_set(em, when);

	/* Without this, when seeking left when it is stopped at EOF,
	 * it starts playing immediately with libefl-1.26.3 even though
	 * emotion_object_play_get() says that it is paused.
	 */
	if (playing == STOPPED) {
	    emotion_object_play_set(em, EINA_TRUE);
	    emotion_object_play_set(em, EINA_FALSE);
	}
#endif
#if SDL_AUDIO
	sdl_start = round(when * sr);
#endif
    }
    set_real_start_time(when);
}

/* Remember at what UTC time the audio file started playing or, rather,
 * when it would have had to start playing from the start
 * if we were to be at "when" seconds through it now.
 */
void
set_real_start_time(secs_t when)
{
    struct timeval tv;

    /* Debugging switch to compare real-time and player-time scrolling */
    if (getenv("SLOPPY") != NULL) { use_real_start_time = FALSE; return; }

    use_real_start_time = (gettimeofday(&tv, NULL) == 0);
    if (!use_real_start_time) {
	perror("Can't get time of day");
    } else {
	real_start_time = tv.tv_sec + tv.tv_usec * 0.000001 - when;
    }
    remember_real_playing_time(when);

    /* This is where we detect "end of file" in mute mode */
    if (mute_mode && when >= audio_file_length() - 1/sr) {
	playing = STOPPED;
    }
}

static void
remember_real_playing_time(secs_t when)
{
    real_playing_time = when;
}

/* Return the audio player's current offset into the audio. */
secs_t
get_playing_time(void)
{
    if (use_real_start_time) {
	struct timeval tv;

	/* Real time keeps on incrementing even if we're not playing */
	if (playing != PLAYING) {
	    return mute_mode ? real_playing_time : get_audio_players_time();
	}

	if (gettimeofday(&tv, NULL) != 0) {
	    perror("Can't get time of day");
	    use_real_start_time = FALSE;
	} else {
	    secs_t now = tv.tv_sec + tv.tv_usec * 0.000001;

	    /* Check whether the audio player has slipped out of sync */
	    if (!mute_mode) {
		secs_t ourtime = now - real_start_time;
		secs_t audio_players_time = get_audio_players_time();

		/* A 16th of a second is noticeable, so resync if it skews
		 * more than a 20th. In practice with SDL2 this happens 
		 * about once a minute. */
#define MAX_SLOP 0.05
		if (DELTA_GT(fabs(ourtime - audio_players_time), MAX_SLOP))
		    real_start_time = now - audio_players_time;
	    }
	    return now - real_start_time;
	}
    }

    fprintf(stdout, "Not using real start time\n");

    /* Fallback: ask the audio player what time they are playing at */
    return get_audio_players_time();
}

/* How far into the piece does the audio playing subsystem think it is? */
secs_t
get_audio_players_time(void)
{
#if EMOTION_AUDIO
    /* Empirically, if its playing, e_o_p_g() returns a value on average
     * 0.181 seconds ahead of what it's actually playing. This used to work
     * for Emotion 1.21 but with 1.25 gets -0.181 for 0 and is jumpy when
     * pausing/continuing.
     */
# if EMOTION_VERSION_MAJOR <= 1 && EMOTION_VERSION_MINOR < 25
    secs_t seek_to = (playing == PLAYING)
		     ? emotion_object_position_get(em) - 0.181
		     : emotion_object_position_get(em);
    return (seek_to < 0.0 ? 0.0 : seek_to);
# else
    return emotion_object_position_get(em);
# endif
#elif SDL_AUDIO
    /* The current playing time is in sdl_start, counted in frames
     * since the start of the piece. */
    if (playing == PLAYING) {
	/* If its playing, we don't know how much of its last buffer it
	 * has already played, but on average it will be half way through. */
	frames_t current = sdl_start - sdl_buffer_size / 2;
	return current < 0 ? (secs_t) 0.0
			   : (secs_t)current / sr;
    }
    else
	return (secs_t)sdl_start / sr;
#endif
}

void
audio_set_softvol(double new)
{
    softvol = new;
    /* With SDL2 we multiply the samples by softvol;
     * EFL does this on its own */
#if EMOTION_AUDIO
    /* Contrary to the documentation, you *can* set it above 1.0.
     * Unfortunately we can't detect when it clips and lower it. */
    emotion_object_audio_volume_set(em, new);
#endif
}

#if SDL_AUDIO
/*
 * SDL audio callback function to fill the buffer at "stream" with
 * "len" bytes of audio data. We assume they want 16-bit ints.
 *
 * SDL seems to ask for 2048 bytes at a time for a 48kHz mono wav file
 * which is (2048/sizeof(short)) / 48000 = 0.0213, about a fiftieth of
 * a second. Presumably, for stereo this would be a hundredth of a second
 * which is close enough for UI purposes.
 */
static void
sdl_fill_audio(void *userdata, Uint8 *stream, int len)
{
    audio_file_t *af = (audio_file_t *)userdata;
    int channels = af->channels;
    frames_t frames_to_read;	/* How many frames are left to read? */
    frames_t frames_read;	/* How many were read from the file? */

    /* SDL has no "playback finished" callback, so spot it here */
    if (sdl_start >= af->frames) {
        stop_playing();
	return;
    }

    frames_to_read = len / (sizeof(short) * channels);
    frames_read = read_cached_audio(af, (char *)stream,
				    af_signed, channels,
				    sdl_start, frames_to_read);
    if (frames_read < 0) {
	/* Happens very occasionally when scrolling vigorously.
	 * Try the real thing. */
	frames_read = read_audio_file(af, (char *)stream,
				      af_signed, channels,
				      sdl_start, frames_to_read);
	if (frames_read < 0) {
	    /* OK, I give up. Send it a blip of silence */
	    fprintf(stdout, "Failed to read %s of cached audio ",
		    frames_to_string(frames_to_read));
	    fprintf(stdout, "at %s for SDL player\n",
		    frames_to_string(sdl_start));
	    bzero(stream, len);
	    sdl_start += frames_to_read;
	    return;
	}
    }
    if (frames_read == 0) {
	/* End of file */
	stop_playing();
	return;
    }

    /* Apply softvol */
    {
	double increase_per_sample = 0.0; /* Init to avoid compiler warning */

	if (softvol_double > 0.0) {
	    /* If we are to double softvol in N seconds, how much should we
	     * multiply it by in each sample period? */
	    increase_per_sample = pow(2.0, 1.0 / (softvol_double * sr));
	}
	if (softvol_double > 0.0 || softvol != 1.0) {
	    int i; short *sp;
	    for (i=0, sp=(short *)stream;
		 i < frames_read * channels;
		 i++, sp++) {
		double value;
		if (softvol_double > 0.0) softvol *= increase_per_sample;
		value = *sp * softvol;
		if (DELTA_LT(value, -32767.0) || DELTA_GT(value, 32767.0)) {
		    /* Reduce softvol to avoid clipping */
		    softvol = 32767.0 / abs(*sp);
		    value = *sp * softvol;
		 }

		/* Plus half a bit of dither? */
		*sp = (short) round(value);
	    }
	}
    }

    if (frames_read >= 0) sdl_start += frames_read;
}
#endif
