% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mann_whitney_test.R
\name{mann_whitney_test}
\alias{mann_whitney_test}
\title{Mann-Whitney test}
\usage{
mann_whitney_test(
  data,
  select = NULL,
  by = NULL,
  weights = NULL,
  mu = 0,
  alternative = "two.sided",
  ...
)
}
\arguments{
\item{data}{A data frame.}

\item{select}{Name(s) of the continuous variable(s) (as character vector)
to be used as samples for the test. \code{select} can be one of the following:
\itemize{
\item \code{select} can be used in combination with \code{by}, in which case \code{select} is
the name of the continous variable (and \code{by} indicates a grouping factor).
\item \code{select} can also be a character vector of length two or more (more than
two names only apply to \code{kruskal_wallis_test()}), in which case the two
continuous variables are treated as samples to be compared. \code{by} must be
\code{NULL} in this case.
\item If \code{select} select is of length \strong{two} and \code{paired = TRUE}, the two samples
are considered as \emph{dependent} and a paired test is carried out.
\item If \code{select} specifies \strong{one} variable and \code{by = NULL}, a one-sample test
is carried out (only applicable for \code{t_test()} and \code{wilcoxon_test()})
\item For \code{chi_squared_test()}, if \code{select} specifies \strong{one} variable and
both \code{by} and \code{probabilities} are \code{NULL}, a one-sample test against given
probabilities is automatically conducted, with equal probabilities for
each level of \code{select}.
}}

\item{by}{Name of the variable indicating the groups. Required if \code{select}
specifies only one variable that contains all samples to be compared in the
test. If \code{by} is not a factor, it will be coerced to a factor. For
\code{chi_squared_test()}, if \code{probabilities} is provided, \code{by} must be \code{NULL}.}

\item{weights}{Name of an (optional) weighting variable to be used for the test.}

\item{mu}{The hypothesized difference in means (for \code{t_test()}) or location
shift (for \code{wilcoxon_test()} and \code{mann_whitney_test()}). The default is 0.}

\item{alternative}{A character string specifying the alternative hypothesis,
must be one of \code{"two.sided"} (default), \code{"greater"} or \code{"less"}. See \code{?t.test}
and \code{?wilcox.test}.}

\item{...}{Additional arguments passed to \code{wilcox.test()} (for unweighted
tests, i.e. when \code{weights = NULL}).}
}
\value{
A data frame with test results. The function returns p and Z-values
as well as effect size r and group-rank-means.
}
\description{
This function performs a Mann-Whitney test (or Wilcoxon rank
sum test for \emph{unpaired} samples). Unlike the underlying base R function
\code{wilcox.test()}, this function allows for weighted tests and automatically
calculates effect sizes. For \emph{paired} (dependent) samples, or for one-sample
tests, please use the \code{wilcoxon_test()} function.

A Mann-Whitney test is a non-parametric test for the null hypothesis that two
\emph{independent} samples have identical continuous distributions. It can be used
for ordinal scales or when the two continuous variables are not normally
distributed. For large samples, or approximately normally distributed variables,
the \code{t_test()} function can be used.
}
\details{
This function is based on \code{\link[=wilcox.test]{wilcox.test()}} and \code{\link[coin:LocationTests]{coin::wilcox_test()}}
(the latter to extract effect sizes). The weighted version of the test is
based on \code{\link[survey:svyranktest]{survey::svyranktest()}}.

Interpretation of the effect size \strong{r}, as a rule-of-thumb:
\itemize{
\item small effect >= 0.1
\item medium effect >= 0.3
\item large effect >= 0.5
}

\strong{r} is calcuated as \eqn{r = \frac{|Z|}{\sqrt{n1 + n2}}}.
}
\section{Which test to use}{

The following table provides an overview of which test to use for different
types of data. The choice of test depends on the scale of the outcome
variable and the number of samples to compare.\tabular{lll}{
   \strong{Samples} \tab \strong{Scale of Outcome} \tab \strong{Significance Test} \cr
   1 \tab binary / nominal \tab \code{chi_squared_test()} \cr
   1 \tab continuous, not normal \tab \code{wilcoxon_test()} \cr
   1 \tab continuous, normal \tab \code{t_test()} \cr
   2, independent \tab binary / nominal \tab \code{chi_squared_test()} \cr
   2, independent \tab continuous, not normal \tab \code{mann_whitney_test()} \cr
   2, independent \tab continuous, normal \tab \code{t_test()} \cr
   2, dependent \tab binary (only 2x2) \tab \code{chi_squared_test(paired=TRUE)} \cr
   2, dependent \tab continuous, not normal \tab \code{wilcoxon_test()} \cr
   2, dependent \tab continuous, normal \tab \code{t_test(paired=TRUE)} \cr
   >2, independent \tab continuous, not normal \tab \code{kruskal_wallis_test()} \cr
   >2, independent \tab continuous,     normal \tab \code{datawizard::means_by_group()} \cr
   >2, dependent \tab continuous, not normal \tab \emph{not yet implemented} (1) \cr
   >2, dependent \tab continuous,     normal \tab \emph{not yet implemented} (2) \cr
}


(1) More than two dependent samples are considered as \emph{repeated measurements}.
For ordinal or not-normally distributed outcomes, these samples are
usually tested using a \code{\link[=friedman.test]{friedman.test()}}, which requires the samples
in one variable, the groups to compare in another variable, and a third
variable indicating the repeated measurements (subject IDs).

(2) More than two dependent samples are considered as \emph{repeated measurements}.
For normally distributed outcomes, these samples are usually tested using
a ANOVA for repeated measurements. A more sophisticated approach would
be using a linear mixed model.
}

\examples{
\dontshow{if (requireNamespace("coin") && requireNamespace("survey")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
data(efc)
# Mann-Whitney-U tests for elder's age by elder's sex.
mann_whitney_test(efc, "e17age", by = "e16sex")
# base R equivalent
wilcox.test(e17age ~ e16sex, data = efc)

# when data is in wide-format, specify all relevant continuous
# variables in `select` and omit `by`
set.seed(123)
wide_data <- data.frame(scale1 = runif(20), scale2 = runif(20))
mann_whitney_test(wide_data, select = c("scale1", "scale2"))
# base R equivalent
wilcox.test(wide_data$scale1, wide_data$scale2)
# same as if we had data in long format, with grouping variable
long_data <- data.frame(
  scales = c(wide_data$scale1, wide_data$scale2),
  groups = as.factor(rep(c("A", "B"), each = 20))
)
mann_whitney_test(long_data, select = "scales", by = "groups")
# base R equivalent
wilcox.test(scales ~ groups, long_data)
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Ben-Shachar, M.S., Patil, I., Thériault, R., Wiernik, B.M.,
Lüdecke, D. (2023). Phi, Fei, Fo, Fum: Effect Sizes for Categorical Data
That Use the Chi‑Squared Statistic. Mathematics, 11, 1982.
\doi{10.3390/math11091982}
\item Bender, R., Lange, S., Ziegler, A. Wichtige Signifikanztests.
Dtsch Med Wochenschr 2007; 132: e24–e25
\item du Prel, J.B., Röhrig, B., Hommel, G., Blettner, M. Auswahl statistischer
Testverfahren. Dtsch Arztebl Int 2010; 107(19): 343–8
}
}
\seealso{
\itemize{
\item \code{\link[=t_test]{t_test()}} for parametric t-tests of dependent and independent samples.
\item \code{\link[=mann_whitney_test]{mann_whitney_test()}} for non-parametric tests of unpaired (independent)
samples.
\item \code{\link[=wilcoxon_test]{wilcoxon_test()}} for Wilcoxon rank sum tests for non-parametric tests
of paired (dependent) samples.
\item \code{\link[=kruskal_wallis_test]{kruskal_wallis_test()}} for non-parametric tests with more than two
independent samples.
\item \code{\link[=chi_squared_test]{chi_squared_test()}} for chi-squared tests (two categorical variables,
dependent and independent).
}
}
