/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     estimator.c                                                    */
/*                                                                          */
/* description:  residual error estimator for elliptic and parabolic        */
/*               problems                                                   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

/*--------------------------------------------------------------------------*/
/*  residual type estimator for quasi linear elliptic problem:              */
/*   -\div A \nabla u + f(.,u,\nabla u) = 0                                 */
/*--------------------------------------------------------------------------*/

struct ellipt_est_data
{
  const DOF_REAL_VEC *uh;
  const BAS_FCTS     *bas_fcts;

  const REAL_D       (*A);
  int                is_diag;
  REAL               (*f)(const EL_INFO *, const QUAD *, int, REAL,
			  const REAL_D);

  const QUAD_FAST  *quad_fast;      /*--  element integration  -------------*/
  const QUAD       *quad;           /*--  face integration     -------------*/

  REAL             *(*rw_est)(EL *);
  REAL             *(*rw_estc)(EL *);

  REAL             *uh_el;          /*--  vector for storing uh on el  -----*/

  FLAGS            f_flag;
  int              norm;            /*--  estimated norm: H1_NORM/L2_NORM --*/

  REAL             C0, C1, C2;

  REAL             est_sum;
  REAL             est_max;
};


static REAL h2_from_det(int dim, REAL det)
{
  FUNCNAME("h2_from_det");

  switch(dim) {
  case 1:
    return det*det;
  case 2:
    return det;
  case 3:
    return pow(det, 2.0/3.0);
  default:
    ERROR_EXIT("Illegal dim!\n");
    return 0.0; /* shut up the compiler */
  }
}

/*--------------------------------------------------------------------------*/
/* element residual:  C0*h_S^2*||-div A nabla u^h + r||_L^2(S)  (H^1)       */
/*                    C0*h_S^4*||-div A nabla u^h + r||_L^2(S)  (L^2)       */
/*--------------------------------------------------------------------------*/

static REAL el_res2(const EL_INFO *el_info, const REAL_D *Lambda, REAL det, 
		    const struct ellipt_est_data *data)
{
  const REAL_DD   *D2uhqp;
  int             dim = el_info->mesh->dim;
  REAL            val, riq = 0.0, h2 = h2_from_det(dim, det), uh_qpi = 0.0;
  int             iq, i, j;
  const REAL      *uh_qp = nil, *grd_uh_qpi = nil;
  const REAL_D    *grd_uh_qp = nil;
  const QUAD_FAST *quad_fast = data->quad_fast;
  const QUAD      *quad = quad_fast->quad;

  if (data->quad_fast->bas_fcts->degree > 1)
    D2uhqp = D2_uh_at_qp(data->quad_fast, Lambda, data->uh_el, nil);
  else
    D2uhqp = nil;

  if (data->f)
  {
    if (data->f_flag & INIT_UH)
      uh_qp = uh_at_qp(quad_fast, data->uh_el, nil);
    else
      uh_qpi = 0.0;

    if (data->f_flag & INIT_GRD_UH)
      grd_uh_qp = grd_uh_at_qp(quad_fast, Lambda, data->uh_el, nil);
    else
      grd_uh_qpi = nil;
  }

  for (val = iq = 0; iq < quad->n_points; iq++)
  {
    if (data->f)
    {
      if (data->f_flag & INIT_UH)     uh_qpi     = uh_qp[iq];
      if (data->f_flag & INIT_GRD_UH) grd_uh_qpi = grd_uh_qp[iq];
      riq = (*data->f)(el_info, quad, iq, uh_qpi, grd_uh_qpi);
    }
    else
      riq = 0.0;

    if (D2uhqp)
    {
      if (data->is_diag)
      {
	for (i = 0; i < DIM_OF_WORLD; i++)
	  riq -= data->A[i][i]*D2uhqp[iq][i][i];
      }
      else
      {
	for (i = 0; i < DIM_OF_WORLD; i++)
	  for (j = 0; j < DIM_OF_WORLD; j++)
	    riq -= data->A[i][j]*D2uhqp[iq][i][j];
      }
    }
    val += quad->w[iq]*SQR(riq);
  }

  if (data->norm == L2_NORM)
    val = data->C0*h2*h2*det*val;
  else
    val = data->C0*h2*det*val;

  return(val);
}


/*--------------------------------------------------------------------------*/
/*  face residuals:  C1*h_Gamma*||[A(u_h)]||_L^2(Gamma)^2          (H^1)    */
/*                   C1*h_S^2*h_Gamma*||[A(u_h)]||_L^2(Gamma)^2    (L^2)    */
/*  Since det_S = det_Gamma*h_Gamma we use for det_Gamma*h_Gamma the term   */
/*  0.5(det_S + det_S')                                                     */
/*--------------------------------------------------------------------------*/

static REAL jump_res2(const EL_INFO *el_info, int face, const REAL_D *Lambda,
		      REAL det, const struct ellipt_est_data *data)
{
  EL_INFO        neigh_info[1];
  int            dim = el_info->mesh->dim;
  int            face_ind_el[dim], face_ind_neigh[dim];
  EL             *neigh = el_info->neigh[face];
  int            opp_v  = el_info->opp_vertex[face];
  int            i, j, i1, i2, iq;
  REAL_D         jump, grd_uh_el, grd_uh_neigh, Lambda_neigh[N_LAMBDA];
  const REAL     *uh_neigh;
  REAL           det_neigh = 0.0, lambda[N_LAMBDA], val = 0.0;
  const BAS_FCTS *bas_fcts = data->uh->fe_space->bas_fcts;
  const QUAD     *quad = data->quad;

/*--------------------------------------------------------------------------*/
/* orient the edge/face => same quadrature nodes from both sides!           */
/*--------------------------------------------------------------------------*/

  sort_wall_indices(dim, el_info->el, face, face_ind_el);
  sort_wall_indices(dim, neigh, opp_v, face_ind_neigh);

  neigh_info->mesh = el_info->mesh;
  neigh_info->el = neigh;
  neigh_info->fill_flag = FILL_COORDS;

  for (j = 0; j < DIM_OF_WORLD; j++)
    neigh_info->coord[opp_v][j] = el_info->opp_coord[face][j];

  for (i = 0; i < dim; i++)
  {
    i1 = face_ind_el[i];
    i2 = face_ind_neigh[i];
    for (j = 0; j < DIM_OF_WORLD; j++)
      neigh_info->coord[i2][j] = el_info->coord[i1][j];
  }

  switch(dim) {
  case 1:
    det_neigh = el_grd_lambda_1d(neigh_info, Lambda_neigh);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    det_neigh = el_grd_lambda_2d(neigh_info, Lambda_neigh);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    det_neigh = el_grd_lambda_3d(neigh_info, Lambda_neigh);
#endif
#endif
  }

  uh_neigh = bas_fcts->get_real_vec(neigh, data->uh, nil);

/*--------------------------------------------------------------------------*/
/*  now eval the jump at all quadrature nodes                               */
/*--------------------------------------------------------------------------*/

  for (val = iq = 0; iq < quad->n_points; iq++)
  {
    lambda[face] = 0.0;
    for (i = 0; i < dim; i++)
      lambda[face_ind_el[i]] = quad->lambda[iq][i];
    eval_grd_uh(lambda, (const REAL_D *)Lambda,
		data->uh_el, bas_fcts, grd_uh_el);

    lambda[opp_v] = 0.0;
    for (i = 0; i < dim; i++)
      lambda[face_ind_neigh[i]] = quad->lambda[iq][i];
    eval_grd_uh(lambda, (const REAL_D *)Lambda_neigh,
		uh_neigh, bas_fcts, grd_uh_neigh);

    if (data->is_diag) 
    {
      for (i = 0; i < DIM_OF_WORLD; i++)
	jump[i] = data->A[i][i]*(grd_uh_el[i] - grd_uh_neigh[i]);
    }
    else
    {
      for (i = 0; i < DIM_OF_WORLD; i++)
	for (jump[i] = j = 0; j < DIM_OF_WORLD; j++)
	  jump[i] += data->A[i][j]*(grd_uh_el[j] - grd_uh_neigh[j]);
    }
    val += quad->w[iq]*SCP_DOW(jump,jump);
  }

  det = 0.5*(det + det_neigh);
  if (data->norm == L2_NORM)
    return(data->C1 * h2_from_det(dim, det) * det * val);
  else
    return(data->C1 * det * val);
}

/*--------------------------------------------------------------------------*/
/*  neuman residual:  C1*h_Gamma*||A(u_h).normal||_L^2(Gamma)^2    (H^1)    */
/*                    C1*h_S^2*h_Gamma*||A(u_h).normal]||_L^2(Gamma)^2 (L^2)*/
/*  Since det_S = det_Gamma*h_Gamma we use for det_Gamma*h_Gamma the term   */
/*  det_S                                                                   */
/*--------------------------------------------------------------------------*/

static REAL neumann_res2(const EL_INFO *el_info, int face, 
			 const REAL_D *Lambda, REAL det, 
			 const struct ellipt_est_data *data)
{
  int            i, j, iq, dim = el_info->mesh->dim;
  REAL           lambda[N_LAMBDA], n_A_grd_uh, val;
  REAL_D         normal, grd_uh, A_grd_uh;
  const QUAD     *quad = data->quad;


  switch(dim) {
  case 1:
    get_wall_normal_1d(el_info, face, normal);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    get_wall_normal_2d(el_info, face, normal);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    get_wall_normal_3d(el_info, face, normal);
#endif
#endif
  }

  for (val = iq = 0; iq < quad->n_points; iq++)
  {
    for (i = 0; i < face; i++)
      lambda[i] = quad->lambda[iq][i];
    lambda[face] = 0.0;
    for (i = face+1; i < dim+1; i++)
      lambda[i] = quad->lambda[iq][i-1];

    eval_grd_uh(lambda, Lambda, data->uh_el, data->bas_fcts, grd_uh);

    if (data->is_diag) 
    {
      for (i = 0; i < DIM_OF_WORLD; i++)
	A_grd_uh[i] = data->A[i][i]*grd_uh[i];
    }
    else
    {
      for (i = 0; i < DIM_OF_WORLD; i++)
	for (A_grd_uh[i] = j = 0; j < DIM_OF_WORLD; j++)
	  A_grd_uh[i] += data->A[i][j]*grd_uh[j];
    }
    n_A_grd_uh = SCP_DOW(normal, A_grd_uh);
    val += quad->w[iq]*SQR(n_A_grd_uh);
  }

  if (data->norm == L2_NORM)
    return(data->C1*h2_from_det(dim, det)*det*val);
  else
    return(data->C1*det*val);
}

static int is_diag_matrix(const REAL_DD A)
{
  int i,j ;

  for (i = 0; i < DIM_OF_WORLD; i++)
    for (j = i+1; j < DIM_OF_WORLD; j++)
      if (ABS(A[i][j]) > 1.e-25 || ABS(A[j][i]) > 1.e-25) return(false);
  return(true);
}

static struct ellipt_est_data ell_data[1];

static void clear_indicator_fct(const EL_INFO *el_info, void *data)
{
  el_info->el->mark = 1;
  if (ell_data->rw_est)  *(*ell_data->rw_est)(el_info->el) = 0.0;
  if (ell_data->rw_estc) *(*ell_data->rw_estc)(el_info->el) = 0.0;
  return;
}

static void ellipt_est_fct(const EL_INFO *el_info, void *data)
{
  EL           *el = el_info->el;
  REAL          det = 0.0, est_el;
  REAL_D        Lambda[N_LAMBDA];
  int           face;
  int           dim = el_info->mesh->dim;
  const S_CHAR *bound = 
    ((dim == 3) ? el_info->face_bound : el_info->edge_bound);
  EL           *neigh;


/*--- if rw_est, then there might already be contributions from jumps ------*/
  est_el = ell_data->rw_est ? *(*ell_data->rw_est)(el) : 0.0;

  ell_data->bas_fcts->get_real_vec(el, ell_data->uh, ell_data->uh_el);

  switch(dim) {
  case 1:
    det = el_grd_lambda_1d(el_info, Lambda);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    det = el_grd_lambda_2d(el_info, Lambda);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    det = el_grd_lambda_3d(el_info, Lambda);
#endif
#endif
  }

/*---  element residual  ---------------------------------------------------*/
  if (ell_data->C0)
    est_el += el_res2(el_info, (const REAL_D *)Lambda, det, ell_data);

/*---  face residuals  -----------------------------------------------------*/

  if (dim > 1 && ell_data->C1) {
    for (face = 0; face < N_NEIGH(dim); face++) {
      if ((neigh = el_info->neigh[face])) {
/*--------------------------------------------------------------------------*/
/*  if rw_est is nil, compute jump for both neighbouring elements           */
/*                    only this allows correct computation of est_max       */
/*  if rw_est is not nil, compute jump only once for each edge/face         */
/*                    if neigh->mark: estimate not computed on neighbour!   */
/*  contribution for the element and for neighbour: 0.5*jump!               */
/*--------------------------------------------------------------------------*/
	if (!ell_data->rw_est  ||  neigh->mark)	{
	  REAL est = jump_res2(el_info, face, 
			       (const REAL_D *)Lambda, det, ell_data);

	  est_el += est;
/*--  if rw_est, add neighbour contribution to neigbour indicator  ---------*/
	  if (ell_data->rw_est) *(*ell_data->rw_est)(neigh) += est;
	}
      }
      else if (IS_NEUMANN(bound[face])) {
	est_el += neumann_res2(el_info, face, 
			       (const REAL_D *)Lambda, det, ell_data);
      }
    }
  }


/*--  if rw_est, write indicator to element  -------------------------------*/
  if (ell_data->rw_est) *(*ell_data->rw_est)(el) = est_el;

  ell_data->est_sum += est_el;
  ell_data->est_max = MAX(ell_data->est_max, est_el);

  el_info->el->mark = 0; /*--- all contributions are computed!  ------------*/
  return;
}

/*--------------------------------------------------------------------------*/

REAL ellipt_est(const DOF_REAL_VEC *uh, ADAPT_STAT *adapt,
		REAL *(*rw_est)(EL *), REAL *(*rw_estc)(EL *),
                int degree, int norm, REAL C[3], const REAL_DD A,
                REAL (*f)(const EL_INFO *,const QUAD *,int,REAL,const REAL_D),
		FLAGS f_flag)
{
  FUNCNAME("ellipt_est");
  static WORKSPACE   ws = {0, nil};
  FLAGS              fill_flag;
  const QUAD         *quad;
  int                dim;

  if (!(ell_data->uh = uh)) {
    MSG("no discrete solution; doing nothing\n");
    return(0.0);
  }


  dim = uh->fe_space->mesh->dim;
  
  if(dim > 1) /* We need a vertex index to orient walls. */
    get_vertex_admin(uh->fe_space->mesh);

  TEST_EXIT(!uh->fe_space->mesh->parametric, "Elliptic error estimator is not yet implemented for parametric meshes, sorry.\n");

  ell_data->bas_fcts = uh->fe_space->bas_fcts;
  ell_data->A        = A;

  if (dim == 1)
    ell_data->is_diag  = true;
  else
    ell_data->is_diag  = is_diag_matrix(A);

  ell_data->f        = f;
  ell_data->f_flag   = f_flag;

  if (degree < 0) degree = 2*ell_data->bas_fcts->degree;
  quad = get_quadrature(dim, degree);
  if (ell_data->bas_fcts->degree > 1)
    fill_flag = INIT_PHI|INIT_GRD_PHI|INIT_D2_PHI;
  else
    fill_flag = INIT_PHI|INIT_GRD_PHI;

  ell_data->quad_fast = get_quad_fast(ell_data->bas_fcts, quad, fill_flag);

  if(dim > 1)
    ell_data->quad    = get_quadrature(dim-1, degree);

  ell_data->rw_est    = rw_est;
  ell_data->rw_estc   = rw_estc;

  REALLOC_WORKSPACE(&ws, ell_data->bas_fcts->n_bas_fcts*sizeof(REAL));
  ell_data->uh_el = (REAL *)ws.work;
  
  ell_data->norm  = norm;
  if (C) {
    ell_data->C0    = C[0] > 1.e-25 ? SQR(C[0]) : 0.0;
    if(dim > 1) {
      ell_data->C1    = C[1] > 1.e-25 ? SQR(C[1]) : 0.0;
      ell_data->C2    = C[2] > 1.e-25 ? SQR(C[2]) : 0.0;
    }
  }
  else
    if(dim == 1)
      ell_data->C0 = 1.0;
    else
      ell_data->C0 = ell_data->C1 = ell_data->C2 = 1.0;

  if (rw_est) /*---  clear error indicators   -----------------------------*/
    mesh_traverse(uh->fe_space->mesh, -1, CALL_LEAF_EL, 
		  clear_indicator_fct, nil);

  ell_data->est_sum = ell_data->est_max = 0.0;

  if(dim == 1)
    fill_flag = FILL_COORDS|CALL_LEAF_EL;
  else
    fill_flag = FILL_NEIGH|FILL_COORDS|FILL_OPP_COORDS|FILL_BOUND|CALL_LEAF_EL;

  mesh_traverse(uh->fe_space->mesh, -1, fill_flag, ellipt_est_fct, nil);

  ell_data->est_sum = sqrt(ell_data->est_sum);
  if (adapt)
  {
    adapt->err_sum = ell_data->est_sum;
    adapt->err_max = ell_data->est_max;
  }

  return(ell_data->est_sum);
}

/*--------------------------------------------------------------------------*/
/*                                                                          */
/* error estimator for (nonlinear) heat equation:                           */
/*              u_t - div A grad u + f(x,t,u,grad u) = 0                    */
/*                                                                          */
/* eta_h = C[0]*||h^2 ((U - Uold)/tau - div A grad U + f(x,t,U,grad U))||   */
/*           + C[1]*|||h^1.5 [A grad U]|||                                  */
/* eta_c = C[2]*||(Uold - Uold_coarse)/tau||                                */
/* eta_t = C[3]*||U - Uold||                                                */
/*                                                                          */
/* heat_est() return value is the TIME DISCRETIZATION ESTIMATE, eta_t       */
/*                                                                          */
/*--------------------------------------------------------------------------*/

struct heat_est_data
{
  const DOF_REAL_VEC *uh, *uh_old;
  const BAS_FCTS     *bas_fcts;

  const REAL_D       (*A);
  int                is_diag;
  REAL               (*f)(const EL_INFO *, const QUAD *, int, REAL,
			  REAL, const REAL_D);

  const QUAD_FAST  *quad_fast;      /*--  element integration  -------------*/
  const QUAD       *quad;           /*--  face integration     -------------*/

  REAL             *(*rw_est)(EL *);
  REAL             *(*rw_estc)(EL *);

  REAL             *uh_el;          /*--  vector for storing uh on el  -----*/
  REAL             *uh_old_el;      /*--  vector for storing uh on el  -----*/
  REAL             *uh_qp;          /*--  vector for storing uh at quadpts -*/
  REAL             *uh_old_qp;      /*--  vector for storing uh at quadpts -*/

  REAL             time, timestep;

  REAL             C0, C1, C2, C3;  /*--  interior,jump,coarsen,time coefs -*/
  REAL             est_sum;
  REAL             est_max;
  REAL             est_t_sum;
  FLAGS            f_flag;
};

static struct heat_est_data heat_data[1];

static void heat_clear_indicator_fct(const EL_INFO *el_info, void *data)
{
  el_info->el->mark = 1;
  if (heat_data->rw_est)  *(*heat_data->rw_est)(el_info->el) = 0.0;
  if (heat_data->rw_estc) *(*heat_data->rw_estc)(el_info->el) = 0.0;
  return;
}

/*--------------------------------------------------------------------------*/
/* element residual:  C0*h_S^2*|| U_t -div A nabla u^h + r ||_L^2(S)        */
/*--------------------------------------------------------------------------*/
/* time residual:  C3*|| U - Uold ||_L^2(S)                                 */
/*--------------------------------------------------------------------------*/

static REAL heat_el_res2(const EL_INFO *el_info, const REAL_D *Lambda,
			 REAL det, struct heat_est_data *data)
{
  const REAL_DD   *D2uhqp;
  int             dim = el_info->mesh->dim;
  REAL            val, riq = 0.0, h2 = h2_from_det(dim, det), uh_qpi = 0.0;
  int             iq, i, j;
  const REAL      *uh_qp = nil, *uh_old_qp = nil, *grd_uh_qpi = nil;
  const REAL_D    *grd_uh_qp = nil;
  const QUAD_FAST *quad_fast = data->quad_fast;
  const QUAD      *quad = quad_fast->quad;

  uh_old_qp = uh_at_qp(quad_fast, data->uh_old_el, data->uh_old_qp);
  uh_qp     = uh_at_qp(quad_fast, data->uh_el, data->uh_qp);

  if (data->C3) {
    for (val = iq = 0; iq < quad->n_points; iq++)
    {
      riq = (uh_qp[iq] - uh_old_qp[iq]);
      val += quad->w[iq]*SQR(riq);
    }
    data->est_t_sum += data->C3*det*val;
  }

  if (!(data->C0)) return(0.0);

  if (data->quad_fast->bas_fcts->degree > 1)
    D2uhqp = D2_uh_at_qp(data->quad_fast, Lambda, data->uh_el, nil);
  else
    D2uhqp = nil;

  if (data->f) {
    if (!(data->f_flag & INIT_UH))
      uh_qpi = 0.0;

    if (data->f_flag & INIT_GRD_UH)
      grd_uh_qp = grd_uh_at_qp(quad_fast, Lambda, data->uh_el, nil);
    else
      grd_uh_qpi = nil;
  }

  for (val = iq = 0; iq < quad->n_points; iq++) {
    riq = (uh_qp[iq] - uh_old_qp[iq]) / data->timestep;

    if (data->f) {
      if (data->f_flag & INIT_UH)     uh_qpi     = uh_qp[iq];
      if (data->f_flag & INIT_GRD_UH) grd_uh_qpi = grd_uh_qp[iq];
      riq += (*data->f)(el_info, quad, iq, data->time, uh_qpi, grd_uh_qpi);
    }

    if (D2uhqp) {
      if (data->is_diag) {
	for (i = 0; i < DIM_OF_WORLD; i++)
	  riq -= data->A[i][i]*D2uhqp[iq][i][i];
      }
      else {
	for (i = 0; i < DIM_OF_WORLD; i++)
	  for (j = 0; j < DIM_OF_WORLD; j++)
	    riq -= data->A[i][j]*D2uhqp[iq][i][j];
      }
    }
    val += quad->w[iq]*SQR(riq);
  }

  val = data->C0*h2*h2*det*val;

  return(val);
}

/*--------------------------------------------------------------------------*/
/*  face residuals:  C1*h_Gamma*||[A(u_h)]||_L^2(Gamma)^2                   */
/*  Since det_S = det_Gamma*h_Gamma we use for det_Gamma*h_Gamma the term   */
/*  0.5(det_S + det_S')                                                     */
/*--------------------------------------------------------------------------*/

static REAL heat_jump_res2(const EL_INFO *el_info, int face,
			   const REAL_D *Lambda,
			   REAL det, const struct heat_est_data *data)
{
  EL_INFO        neigh_info[1];
  int            dim = el_info->mesh->dim;
  int            face_ind_el[dim], face_ind_neigh[dim];
  EL             *neigh = el_info->neigh[face];
  int            opp_v  = el_info->opp_vertex[face];
  int            i, j, i1, i2, iq;
  REAL_D         jump, grd_uh_el, grd_uh_neigh, Lambda_neigh[N_LAMBDA];
  const REAL     *uh_neigh;
  REAL           det_neigh = 0.0, lambda[N_LAMBDA], val = 0.0;
  const BAS_FCTS *bas_fcts = data->uh->fe_space->bas_fcts;
  const QUAD     *quad = data->quad;

/*--------------------------------------------------------------------------*/
/* orient the edge/face => same quadrature nodes from both sides!           */
/*--------------------------------------------------------------------------*/

  sort_wall_indices(dim, el_info->el, face, face_ind_el);
  sort_wall_indices(dim, neigh, opp_v, face_ind_neigh);

  neigh_info->mesh = el_info->mesh;
  neigh_info->el = neigh;
  neigh_info->fill_flag = FILL_COORDS;

  for (j = 0; j < DIM_OF_WORLD; j++)
    neigh_info->coord[opp_v][j] = el_info->opp_coord[face][j];

  for (i = 0; i < dim; i++)
  {
    i1 = face_ind_el[i];
    i2 = face_ind_neigh[i];
    for (j = 0; j < DIM_OF_WORLD; j++)
      neigh_info->coord[i2][j] = el_info->coord[i1][j];
  }

  switch(dim) {
  case 1:
    det_neigh = el_grd_lambda_1d(neigh_info, Lambda_neigh);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    det_neigh = el_grd_lambda_2d(neigh_info, Lambda_neigh);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    det_neigh = el_grd_lambda_3d(neigh_info, Lambda_neigh);
#endif
#endif
  }

  uh_neigh = bas_fcts->get_real_vec(neigh, data->uh, nil);

/*--------------------------------------------------------------------------*/
/*  now eval the jump at all quadrature nodes                               */
/*--------------------------------------------------------------------------*/

  for (val = iq = 0; iq < quad->n_points; iq++)
  {
    lambda[face] = 0.0;
    for (i = 0; i < dim; i++)
      lambda[face_ind_el[i]] = quad->lambda[iq][i];
    eval_grd_uh(lambda, Lambda, data->uh_el, bas_fcts, grd_uh_el);

    lambda[opp_v] = 0.0;
    for (i = 0; i < dim; i++)
      lambda[face_ind_neigh[i]] = quad->lambda[iq][i];
    eval_grd_uh(lambda, (const REAL_D *)Lambda_neigh, uh_neigh, bas_fcts, 
		grd_uh_neigh);

    if (data->is_diag) 
    {
      for (i = 0; i < DIM_OF_WORLD; i++)
	jump[i] = data->A[i][i]*(grd_uh_el[i] - grd_uh_neigh[i]);
    }
    else
    {
      for (i = 0; i < DIM_OF_WORLD; i++)
	for (jump[i] = j = 0; j < DIM_OF_WORLD; j++)
	  jump[i] += data->A[i][j]*(grd_uh_el[j] - grd_uh_neigh[j]);
    }
    val += quad->w[iq]*SCP_DOW(jump,jump);
  }

  det = 0.5*(det + det_neigh);

  return(data->C1*h2_from_det(dim,det)*det*val);
}

/*--------------------------------------------------------------------------*/
/*  neuman residual:  C1*h_Gamma*||A(u_h).normal||_L^2(Gamma)^2             */
/*  Since det_S = det_Gamma*h_Gamma we use for det_Gamma*h_Gamma the term   */
/*  det_S                                                                   */
/*--------------------------------------------------------------------------*/

static REAL heat_neumann_res2(const EL_INFO *el_info, int face, 
			      const REAL_D *Lambda, REAL det, 
			      const struct heat_est_data *data)
{
  int            i, j, iq, dim = el_info->mesh->dim;
  REAL           lambda[N_LAMBDA], n_A_grd_uh, val;
  REAL_D         normal, grd_uh, A_grd_uh;
  const QUAD     *quad = data->quad;

  switch(dim) {
  case 1:
    get_wall_normal_1d(el_info, face, normal);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    get_wall_normal_2d(el_info, face, normal);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    get_wall_normal_3d(el_info, face, normal);
#endif
#endif
  }

  for (val = iq = 0; iq < quad->n_points; iq++)
  {
    for (i = 0; i < face; i++)
      lambda[i] = quad->lambda[iq][i];
    lambda[face] = 0.0;
    for (i = face+1; i < dim+1; i++)
      lambda[i] = quad->lambda[iq][i-1];

    eval_grd_uh(lambda, Lambda, data->uh_el, data->bas_fcts, grd_uh);

    if (data->is_diag) 
    {
      for (i = 0; i < DIM_OF_WORLD; i++)
	A_grd_uh[i] = data->A[i][i]*grd_uh[i];
    }
    else
    {
      for (i = 0; i < DIM_OF_WORLD; i++)
	for (A_grd_uh[i] = j = 0; j < DIM_OF_WORLD; j++)
	  A_grd_uh[i] += data->A[i][j]*grd_uh[j];
    }
    n_A_grd_uh = SCP_DOW(normal, A_grd_uh);
    val += quad->w[iq]*SQR(n_A_grd_uh);
  }

  return(data->C1*h2_from_det(dim, det)*det*val);
}

/*--------------------------------------------------------------------------*/

static void heat_est_fct(const EL_INFO *el_info, void *data)
{
  EL           *el = el_info->el;
  REAL          det = 0.0, est_el;
  REAL_D        Lambda[N_LAMBDA];
  int           dim = el_info->mesh->dim;
  const S_CHAR *bound = 
    ((dim == 3) ? el_info->face_bound : el_info->edge_bound);
  int           face;
  EL           *neigh;

/*--- if rw_est, then there might already be contributions from jumps ------*/
  est_el = heat_data->rw_est ? *(*heat_data->rw_est)(el) : 0.0;

  heat_data->bas_fcts->get_real_vec(el, heat_data->uh,
				    heat_data->uh_el);
  heat_data->bas_fcts->get_real_vec(el, heat_data->uh_old,
				    heat_data->uh_old_el);

  switch(dim) {
  case 1:
    det = el_grd_lambda_1d(el_info, Lambda);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    det = el_grd_lambda_2d(el_info, Lambda);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    det = el_grd_lambda_3d(el_info, Lambda);
#endif
#endif
  }

/*---  element and time residual  ------------------------------------------*/
  if (heat_data->C0 || heat_data->C3)
    est_el += heat_el_res2(el_info, (const REAL_D *)Lambda, det, heat_data);

/*---  face residuals  -----------------------------------------------------*/

  if (dim > 1 && heat_data->C1) {
    for (face = 0; face < N_NEIGH(dim); face++)
    {
      if ((neigh = el_info->neigh[face]))
      {
/*--------------------------------------------------------------------------*/
/*  if rw_est is nil, compute jump for both neighbouring elements           */
/*                    only this allows correct computation of est_max       */
/*  if rw_est is not nil, compute jump only once for each edge/face         */
/*                    if neigh->mark: estimate not computed on neighbour!   */
/*  contribution for the element and for neighbour: 0.5*jump!               */
/*--------------------------------------------------------------------------*/
	if (!heat_data->rw_est  ||  neigh->mark)
	{
	  REAL est = heat_jump_res2(el_info, face, 
				    (const REAL_D *)Lambda, det, heat_data);

	  est_el += est;
/*--  if rw_est, add neighbour contribution to neigbour indicator  ---------*/
	  if (heat_data->rw_est) *(*heat_data->rw_est)(neigh) += est;
	}
      }
      else if (IS_NEUMANN(bound[face]))
      {
	est_el += heat_neumann_res2(el_info, face, 
				    (const REAL_D *)Lambda, det, heat_data);
      }
    }
  }


#if 0
/*--  if rw_estc, calculate coarsening error estimate  ---------------------*/
  if (heat_data->rw_estc && heat_data->C2)
  {
    *(*heat_data->rw_estc)(el) = heat_estc_el(el_info, det, heat_data);
  }
#endif

/*--  if rw_est, write indicator to element  -------------------------------*/
  if (heat_data->rw_est) *(*heat_data->rw_est)(el) = est_el;

  heat_data->est_sum += est_el;
  heat_data->est_max = MAX(heat_data->est_max, est_el);

  el_info->el->mark = 0; /*--- all contributions are computed!  ------------*/

  return;
}

/*--------------------------------------------------------------------------*/

REAL heat_est(const DOF_REAL_VEC *uh, ADAPT_INSTAT *adapt,
	      REAL *(*rw_est)(EL *), REAL *(*rw_estc)(EL *),
	      int degree, REAL C[4], 
	      const DOF_REAL_VEC *uh_old, const REAL_DD A,
	      REAL (*f)(const EL_INFO *, const QUAD *, int iq, REAL t,
			REAL u, const REAL_D grd_u),
	      FLAGS f_flag)
{
  FUNCNAME("heat_est");
  int                dim;
  static WORKSPACE   ws = {0, nil};
  FLAGS              fill_flag;
  const QUAD         *quad;

  if (!(heat_data->uh = uh))
  {
    MSG("no discrete solution; doing nothing\n");
    return(0.0);
  }

  dim = uh->fe_space->mesh->dim;

  if (!(heat_data->uh_old = uh_old))
  {
    MSG("no discrete solution from previous timestep; doing nothing\n");
    /* here, initial error could be calculated... */
    return(0.0);
  }

  TEST_EXIT(!uh->fe_space->mesh->parametric, "Parabolic error estimator is not yet implemented for parametric meshes, sorry.\n");

  if(dim > 1) /* We need a vertex index to orient walls. */
    get_vertex_admin(uh->fe_space->mesh);

  heat_data->bas_fcts = uh->fe_space->bas_fcts;
  heat_data->A        = A;
  if(dim == 1)
    heat_data->is_diag  = true;
  else
    heat_data->is_diag  = is_diag_matrix(A);

  heat_data->f        = f;
  heat_data->f_flag   = f_flag;

  if (degree < 0) degree = 2*heat_data->bas_fcts->degree;
  quad = get_quadrature(dim, degree);
  if (heat_data->bas_fcts->degree > 1)
    fill_flag = INIT_PHI|INIT_GRD_PHI|INIT_D2_PHI;
  else
    fill_flag = INIT_PHI|INIT_GRD_PHI;

  heat_data->quad_fast = get_quad_fast(heat_data->bas_fcts, quad, fill_flag);
  if (dim > 1)
    heat_data->quad      = get_quadrature(dim-1, degree);

  heat_data->rw_est    = rw_est;
  heat_data->rw_estc   = rw_estc;

  REALLOC_WORKSPACE(&ws, 2*(heat_data->bas_fcts->n_bas_fcts
			    + heat_data->quad_fast->n_points)*sizeof(REAL));
  heat_data->uh_el     = (REAL *)ws.work;
  heat_data->uh_old_el = heat_data->uh_el + heat_data->bas_fcts->n_bas_fcts;
  heat_data->uh_qp     =
    heat_data->uh_old_el + heat_data->bas_fcts->n_bas_fcts;
  heat_data->uh_old_qp = heat_data->uh_qp + heat_data->quad_fast->n_points;

#if 0
  heat_data->uh_qp     = MEM_ALLOC(6, REAL);
  heat_data->uh_old_qp = MEM_ALLOC(6, REAL);
#endif

  
  if (C)
  {
    heat_data->C0    = C[0] > 1.e-25 ? SQR(C[0]) : 0.0;
    heat_data->C1    = C[1] > 1.e-25 ? SQR(C[1]) : 0.0;
    heat_data->C2    = C[2] > 1.e-25 ? SQR(C[2]) : 0.0;
    heat_data->C3    = C[3] > 1.e-25 ? SQR(C[3]) : 0.0;
  }
  else
  {
    heat_data->C0 = heat_data->C1 = heat_data->C2 = heat_data->C3 = 1.0;
  }

  heat_data->time = adapt->time;
  heat_data->timestep = adapt->timestep;


  if (rw_est)  /*---  clear error indicators   -----------------------------*/
    mesh_traverse(uh->fe_space->mesh, -1, CALL_LEAF_EL,
		  heat_clear_indicator_fct, nil);

  heat_data->est_sum = heat_data->est_max = heat_data->est_t_sum = 0.0;
  if(dim == 1)
    fill_flag = FILL_COORDS|CALL_LEAF_EL;
  else
    fill_flag = FILL_NEIGH|FILL_COORDS|FILL_OPP_COORDS|FILL_BOUND|CALL_LEAF_EL;

  mesh_traverse(uh->fe_space->mesh, -1, fill_flag, heat_est_fct, nil);

  heat_data->est_sum   = sqrt(heat_data->est_sum);
  heat_data->est_t_sum = sqrt(heat_data->est_t_sum);
  if (adapt)
  {
    adapt->adapt_space->err_sum = heat_data->est_sum;
    adapt->adapt_space->err_max = heat_data->est_max;
/*     adapt->err_t = heat_data->est_t_sum; */
  }

  return(heat_data->est_t_sum);
}
