/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     Common/refine.c                                                */
/*                                                                          */
/*                                                                          */
/* description:  Common refinement routines shared among all dimensions     */
/*               This file contains refine_?d.c for ? = 1,2,3 !!            */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Univesitaet Bremen                                           */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"
#include "alberta_intern.h"

/*--------------------------------------------------------------------------*/
/*  refinement routines for global refinement:				    */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*  global_refine_fct:							    */
/*  sets the mark on all elements that have to be refined                   */
/*--------------------------------------------------------------------------*/

typedef struct refine_traverse_data {
  int global_refine_mark;
} REFINE_TRAVERSE_DATA;

static void global_refine_fct(const EL_INFO *el_info, void *data)
{
  el_info->el->mark = ((REFINE_TRAVERSE_DATA *)data)->global_refine_mark;
}

/*--------------------------------------------------------------------------*/
/*  global_refine:							    */
/*  refines every element of mesh at least mark times			    */
/*--------------------------------------------------------------------------*/

U_CHAR global_refine(MESH *mesh, int mark)
{
  REFINE_TRAVERSE_DATA td[1] = {{0}};
  if (mark <= 0)  return(0);
  td->global_refine_mark = mark;
  mesh_traverse(mesh, -1, CALL_LEAF_EL, global_refine_fct, td);
  return(refine(mesh));
}


DOF_VEC_LIST *AI_get_dof_vec_list(MESH *mesh)
{
  MESH_MEM_INFO  *mem_info = (MESH_MEM_INFO *)mesh->mem_info;

  if(!mem_info->dvlist)
    mem_info->dvlist = MEM_CALLOC(1, DOF_VEC_LIST);

  return mem_info->dvlist;
}

void AI_free_dof_vec_list(MESH *mesh)
{
  MESH_MEM_INFO  *mem_info = (MESH_MEM_INFO *)mesh->mem_info;

  if(mem_info->dvlist) {
    MEM_FREE(mem_info->dvlist, 1, DOF_VEC_LIST);
    mem_info->dvlist = nil;
  }
}

static int count_refine_interpol(MESH *mesh)
{
  FUNCNAME("count_refine_interpol");
  DOF_VEC_LIST    *dvlist;
  DOF_ADMIN       *admin;
  int              iadmin;
  DOF_REAL_VEC    *drv;
  DOF_REAL_D_VEC  *drdv;
  DOF_INT_VEC     *div;
  DOF_DOF_VEC     *ddv;
  DOF_UCHAR_VEC   *duv;
  DOF_SCHAR_VEC   *dsv;
  DOF_PTR_VEC     *dpv;
  DOF_MATRIX      *dm;
  DOF_DOWB_MATRIX *ddm;
  int              nri = 0;
  int              n_dof_int_vec=0,  n_dof_dof_vec=0, n_dof_uchar_vec=0,
                   n_dof_schar_vec=0, n_dof_real_vec=0, n_dof_real_d_vec=0,
                   n_dof_ptr_vec=0, n_dof_matrix=0, n_dof_dowb_matrix=0;

  dvlist = AI_get_dof_vec_list(mesh);

  for (iadmin = 0; iadmin < mesh->n_dof_admin; iadmin++)
  {
    admin = mesh->dof_admin[iadmin];

    for (div = admin->dof_int_vec; div; div = div->next)
    {
      if (div->refine_interpol) n_dof_int_vec++;
    }

    for (ddv = admin->dof_dof_vec; ddv; ddv = ddv->next)
    {
      if (ddv->refine_interpol) n_dof_dof_vec++;
    }

    for (ddv = admin->int_dof_vec; ddv; ddv = ddv->next)
    {
      if (ddv->refine_interpol) n_dof_dof_vec++;
    }

    for (duv = admin->dof_uchar_vec; duv; duv = duv->next)
    {
      if (duv->refine_interpol) n_dof_uchar_vec++;
    }

    for (dsv = admin->dof_schar_vec; dsv; dsv = dsv->next)
    {
      if (dsv->refine_interpol) n_dof_schar_vec++;
    }

    for (drv = admin->dof_real_vec; drv; drv = drv->next)
    {
      if (drv->refine_interpol) n_dof_real_vec++;
    }

    for (dpv = admin->dof_ptr_vec; dpv; dpv = dpv->next)
    {
      if (dpv->refine_interpol) n_dof_ptr_vec++;
    }

    for (drdv = admin->dof_real_d_vec; drdv; drdv = drdv->next)
    {
      if (drdv->refine_interpol) n_dof_real_d_vec++;
    }

    for (dm = admin->dof_matrix; dm; dm = dm->next)
    {
      if (dm->refine_interpol) n_dof_matrix++;
    }

    for (ddm = admin->dof_dowb_matrix; ddm; ddm = ddm->next)
    {
      if (ddm->refine_interpol) n_dof_dowb_matrix++;
    }
  }

  nri = n_dof_int_vec + n_dof_dof_vec + n_dof_uchar_vec + n_dof_schar_vec
    + n_dof_real_vec + n_dof_ptr_vec + n_dof_real_d_vec + n_dof_matrix 
    + n_dof_dowb_matrix;

  if (nri > 0)
  {
    if (dvlist->size < nri) {
      dvlist->list = MEM_REALLOC(dvlist->list, dvlist->size, nri+5, void *);
      dvlist->size = nri+5;
    }

    nri = 0;

    if (n_dof_int_vec)
      dvlist->dof_int_vec    = (DOF_INT_VEC **)(dvlist->list+nri);
    else
      dvlist->dof_int_vec    = nil;
    nri += n_dof_int_vec;

    if (n_dof_dof_vec)
      dvlist->dof_dof_vec    = (DOF_DOF_VEC **)(dvlist->list+nri);
    else
      dvlist->dof_dof_vec    = nil;
    nri += n_dof_dof_vec;

    if (n_dof_uchar_vec)
      dvlist->dof_uchar_vec  = (DOF_UCHAR_VEC **)(dvlist->list+nri);
    else
      dvlist->dof_uchar_vec  = nil;
    nri += n_dof_uchar_vec;

    if (n_dof_schar_vec)
      dvlist->dof_schar_vec  = (DOF_SCHAR_VEC **)(dvlist->list+nri);
    else
      dvlist->dof_schar_vec  = nil;
    nri += n_dof_schar_vec;

    if (n_dof_real_vec)
      dvlist->dof_real_vec   = (DOF_REAL_VEC **)(dvlist->list+nri);
    else
      dvlist->dof_real_vec   = nil;
    nri += n_dof_real_vec;

    if (n_dof_ptr_vec)
      dvlist->dof_ptr_vec    = (DOF_PTR_VEC **)(dvlist->list+nri);
    else
      dvlist->dof_ptr_vec    = nil;
    nri += n_dof_ptr_vec;

    if (n_dof_real_d_vec)
      dvlist->dof_real_d_vec = (DOF_REAL_D_VEC **)(dvlist->list+nri);
    else
      dvlist->dof_real_d_vec = nil;
    nri += n_dof_real_d_vec;

    if (n_dof_matrix)
      dvlist->dof_matrix     = (DOF_MATRIX **)(dvlist->list+nri);
    else
      dvlist->dof_matrix     = nil;
    nri += n_dof_matrix;

    if (n_dof_dowb_matrix)
      dvlist->dof_dowb_matrix= (DOF_DOWB_MATRIX **)(dvlist->list+nri);
    else
      dvlist->dof_dowb_matrix     = nil;
    nri += n_dof_dowb_matrix;
    DEBUG_TEST_EXIT(nri <= dvlist->size, "error in dvlist->size");

    dvlist->n_dof_int_vec     = 0;
    dvlist->n_dof_dof_vec     = 0;
    dvlist->n_dof_uchar_vec   = 0;
    dvlist->n_dof_schar_vec   = 0;
    dvlist->n_dof_real_vec    = 0;
    dvlist->n_dof_real_d_vec  = 0;
    dvlist->n_dof_ptr_vec     = 0;
    dvlist->n_dof_matrix      = 0;
    dvlist->n_dof_dowb_matrix = 0;

    for (iadmin = 0; iadmin < mesh->n_dof_admin; iadmin++)
    {
      admin = mesh->dof_admin[iadmin];
      for (div = admin->dof_int_vec; div; div = div->next)
      {
	if (div->refine_interpol) 
	  dvlist->dof_int_vec[dvlist->n_dof_int_vec++] = div;
      }
      for (ddv = admin->dof_dof_vec; ddv; ddv = ddv->next) 
      {
	if (ddv->refine_interpol) 
	  dvlist->dof_dof_vec[dvlist->n_dof_dof_vec++] = ddv;
      }
      for (ddv = admin->int_dof_vec; ddv; ddv = ddv->next) 
      {
	if (ddv->refine_interpol) 
	  dvlist->dof_dof_vec[dvlist->n_dof_dof_vec++] = ddv;
      }
      for (duv = admin->dof_uchar_vec; duv; duv = duv->next) 
      {
	if (duv->refine_interpol)
	  dvlist->dof_uchar_vec[dvlist->n_dof_uchar_vec++] = duv;
      }
      for (dsv = admin->dof_schar_vec; dsv; dsv = dsv->next) 
      {
	if (dsv->refine_interpol) 
	  dvlist->dof_schar_vec[dvlist->n_dof_schar_vec++] = dsv;
      }
      for (drv = admin->dof_real_vec; drv; drv = drv->next) 
      {
	if (drv->refine_interpol)
	  dvlist->dof_real_vec[dvlist->n_dof_real_vec++] = drv;
      }
      for (drdv = admin->dof_real_d_vec; drdv; drdv = drdv->next) 
      {
	if (drdv->refine_interpol) 
	  dvlist->dof_real_d_vec[dvlist->n_dof_real_d_vec++] = drdv;
      }
      for (dpv = admin->dof_ptr_vec; dpv; dpv = dpv->next) 
      {
	if (dpv->refine_interpol)
	  dvlist->dof_ptr_vec[dvlist->n_dof_ptr_vec++] = dpv;
      }
      for (dm = admin->dof_matrix; dm; dm = dm->next) 
      {
	if (dm->refine_interpol) 
	  dvlist->dof_matrix[dvlist->n_dof_matrix++] = dm;
      }
      for (ddm = admin->dof_dowb_matrix; ddm; ddm = ddm->next) 
      {
	if (ddm->refine_interpol) 
	  dvlist->dof_dowb_matrix[dvlist->n_dof_dowb_matrix++] = ddm;
      }
    }

    DEBUG_TEST_EXIT(dvlist->n_dof_int_vec == n_dof_int_vec,
		"error in n_dof_int_vec");
    DEBUG_TEST_EXIT(dvlist->n_dof_dof_vec == n_dof_dof_vec,
		"error in n_dof_dof_vec");
    DEBUG_TEST_EXIT(dvlist->n_dof_uchar_vec == n_dof_uchar_vec,
		"error in n_dof_uchar_vec");
    DEBUG_TEST_EXIT(dvlist->n_dof_schar_vec == n_dof_schar_vec,
		"error in n_dof_schar_vec");
    DEBUG_TEST_EXIT(dvlist->n_dof_real_vec == n_dof_real_vec,
		"error in n_dof_real_vec");
    DEBUG_TEST_EXIT(dvlist->n_dof_real_d_vec == n_dof_real_d_vec,
		"error in n_dof_real_d_vec");
    DEBUG_TEST_EXIT(dvlist->n_dof_ptr_vec == n_dof_ptr_vec,
		"error in n_dof_ptr_vec");
    DEBUG_TEST_EXIT(dvlist->n_dof_matrix == n_dof_matrix,
		"error in n_dof_matrix");
    DEBUG_TEST_EXIT(dvlist->n_dof_dowb_matrix == n_dof_dowb_matrix,
		"error in n_dof_dowb_matrix");
  }
  else
  {
    dvlist->dof_int_vec     = nil;
    dvlist->dof_dof_vec     = nil;
    dvlist->dof_uchar_vec   = nil;
    dvlist->dof_schar_vec   = nil;
    dvlist->dof_real_vec    = nil;
    dvlist->dof_real_d_vec  = nil;
    dvlist->dof_ptr_vec     = nil;
    dvlist->dof_matrix      = nil;
    dvlist->dof_dowb_matrix = nil;
  }

  return(nri);
}


static void refine_interpol(MESH *mesh, RC_LIST_EL *list, int n_el)
{
  DOF_VEC_LIST    *dvlist;
  DOF_REAL_VEC    *drv;
  DOF_REAL_D_VEC  *drdv;
  DOF_INT_VEC     *div;
  DOF_DOF_VEC     *ddv;
  DOF_UCHAR_VEC   *duv;
  DOF_SCHAR_VEC   *dsv;
  DOF_PTR_VEC     *dpv;
  DOF_MATRIX      *dm;
  DOF_DOWB_MATRIX *ddm;
  int              i;

  dvlist = AI_get_dof_vec_list(mesh);

  for (i = 0; i < dvlist->n_dof_ptr_vec; i++) 
  {
    dpv = dvlist->dof_ptr_vec[i];
    dpv->refine_interpol(dpv, list, n_el);
  }

  for (i = 0; i < dvlist->n_dof_int_vec; i++) 
  {
    div = dvlist->dof_int_vec[i];
    div->refine_interpol(div, list, n_el);
  }

  for (i = 0; i < dvlist->n_dof_dof_vec; i++) 
  {
    ddv = dvlist->dof_dof_vec[i];
    ddv->refine_interpol(ddv, list, n_el);
  }

  for (i = 0; i < dvlist->n_dof_uchar_vec; i++) 
  {
    duv = dvlist->dof_uchar_vec[i];
    duv->refine_interpol(duv, list, n_el);
  }

  for (i = 0; i < dvlist->n_dof_schar_vec; i++)
  {
    dsv = dvlist->dof_schar_vec[i];
    dsv->refine_interpol(dsv, list, n_el);
  }

  for (i = 0; i < dvlist->n_dof_real_vec; i++) 
  {
    drv = dvlist->dof_real_vec[i];
    drv->refine_interpol(drv, list, n_el);
  }

  for (i = 0; i < dvlist->n_dof_real_d_vec; i++) 
  {
    drdv = dvlist->dof_real_d_vec[i];
    drdv->refine_interpol(drdv, list, n_el);
  }

  for (i = 0; i < dvlist->n_dof_matrix; i++)
  {
    dm = dvlist->dof_matrix[i];
    dm->refine_interpol(dm, list, n_el);
  }

  for (i = 0; i < dvlist->n_dof_dowb_matrix; i++)
  {
    ddm = dvlist->dof_dowb_matrix[i];
    ddm->refine_interpol(ddm, list, n_el);
  }

  return;
}

static int             do_more_refine_1d = 0; 
static int             call_refine_interpol_1d = 0;

#include "refine_1d.c"

#if DIM_OF_WORLD > 1
static int             do_more_refine_2d = 0; 
static int             call_refine_interpol_2d = 0;

static TRAVERSE_STACK *stack_2d;
#include "refine_2d.c"

#if DIM_OF_WORLD > 2
static int             do_more_refine_3d = 0; 
static int             call_refine_interpol_3d = 0;

static TRAVERSE_STACK *stack_3d;
static RC_LIST_EL *static_ref_list_3d = nil;

#include "refine_3d.c"
#endif
#endif

/*--------------------------------------------------------------------------*/
/*  refine:     							    */
/*  traversal routine for recursive refinement of a triangulation; basic    */
/*  refinement module; used by all other refinement routines like           */
/*  global_refine()                                                         */
/*--------------------------------------------------------------------------*/

static void transfer_fct(const EL_INFO *elinfo, void *data)
{
  MESH            *mesh = elinfo->mesh; 
  EL              *s_el  = elinfo->el,
                  *m_el;
  DOF_PTR_VEC     *master_binding;
  const DOF_ADMIN *s_admin;

  if(s_el->mark > 0) { 
    master_binding = ((MESH_MEM_INFO *)mesh->mem_info)->master_binding;
    s_admin = master_binding->fe_space->admin;
    
    m_el = (EL *)master_binding->vec[s_el->dof[mesh->node[CENTER]]
				     [s_admin->n0_dof[CENTER]]];
    
    m_el->mark = MAX(m_el->mark, 1);
  }

  return;
}

U_CHAR refine(MESH *mesh)
{
  FUNCNAME("refine");
  MESH_MEM_INFO *mem_info = ((MESH_MEM_INFO *)mesh->mem_info);
  U_CHAR         mesh_refined;

  if(mem_info->n_slaves) {
/****************************************************************************/
/* We are on a master mesh.                                                 */
/****************************************************************************/

#if DIM_OF_WORLD > 1
/****************************************************************************/
/* Check if we have an entire hierarchy of meshes. In this case, we need    */
/* to set call_refine_interpol_?d to nonzero to make the triple refinement  */
/* work!                                                                    */
/****************************************************************************/
    if(mesh->dim == 2) {
      int i;

      call_refine_interpol_1d = 0;

      for(i = 0; i < mem_info->n_slaves; i++)
	call_refine_interpol_1d +=
	  count_refine_interpol(mem_info->slaves[i]);
    }

#if DIM_OF_WORLD == 3
    if(mesh->dim == 3) {
      int i, j;

      call_refine_interpol_1d = 0;
      call_refine_interpol_2d = 0;

      for(i = 0; i < mem_info->n_slaves; i++) {
	call_refine_interpol_2d +=
	  count_refine_interpol(mem_info->slaves[i]);

	if(((MESH_MEM_INFO *)(mem_info->slaves[i]->mem_info))->n_slaves) {
	  MESH_MEM_INFO *mem_info_2 =
	    ((MESH_MEM_INFO *)mem_info->slaves[i]->mem_info);
	  

	  for(j = 0; j < mem_info_2->n_slaves; j++)
	    call_refine_interpol_1d +=
	      count_refine_interpol(mem_info_2->slaves[j]);
	}
      }
    }
#endif
#endif
  }

  if(mem_info->master) {
/****************************************************************************/
/* We are on a slave mesh.                                                  */
/****************************************************************************/
    int    n_slave_elements = mesh->n_elements;

/* Transfer the refinement marks to the master mesh.                        */
    do {
      mesh_traverse(mesh, 0, FILL_NOTHING|CALL_LEAF_EL, transfer_fct, nil);
      mesh_refined = refine(mem_info->master);
    } while (mesh_refined);
    return (mesh->n_elements > n_slave_elements) ? MESH_REFINED : 0; 
  }
  else {
/****************************************************************************/
/* We are on a top level master mesh.                                       */
/****************************************************************************/

    /* Advance cookies on this mesh and all its slaves. */
    AI_advance_cookies_rec(mesh);
  }

  switch(mesh->dim) {
  case 0:
    WARNING("No refinement possible for dim==0!\n");
    return 0;
    break;
  case 1:
    mesh_refined = refine_1d(mesh);

    if(mesh_refined && !mesh->parametric)
      AI_post_refine_1d(mesh);

    return mesh_refined;
    break;
#if DIM_OF_WORLD > 1
  case 2:
    mesh_refined = refine_2d(mesh);

    if(mesh_refined && !mesh->parametric) {
      int i;

      AI_post_refine_2d(mesh);

      for(i = 0; i < mem_info->n_slaves; i++)
	AI_post_refine_1d(mem_info->slaves[i]);
    }

    return mesh_refined;
    break;
#if DIM_OF_WORLD > 2
  case 3:
    mesh_refined = refine_3d(mesh);

    if(mesh_refined && !mesh->parametric) {
      int i, j;

      post_refine_3d(mesh);

      for(i = 0; i < mem_info->n_slaves; i++) {
	AI_post_refine_2d(mem_info->slaves[i]);

	if(((MESH_MEM_INFO *)(mem_info->slaves[i]->mem_info))->n_slaves) {
	  MESH_MEM_INFO *mem_info_2 =
	    ((MESH_MEM_INFO *)mem_info->slaves[i]->mem_info);
	  

	  for(j = 0; j < mem_info_2->n_slaves; j++)
	    AI_post_refine_1d(mem_info_2->slaves[j]);
	}
      }
    }

    return mesh_refined;
    break;
#endif
#endif
  default:
    ERROR_EXIT("Illegal dim during refining!\n");
  }
  
  return 0;                                       /* Statement not reached! */
}
