/*
  CoreLinux++ 
  Copyright (C) 1999,2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/

#if   !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

#if   !defined(__ENVIRONMENT_HPP)
#include <Environment.hpp>
#endif

extern "C"
{
   #include <fcntl.h>
   #include <sys/stat.h>
   #include <unistd.h>
   #include <sys/types.h>
   #include <sys/resource.h>
   #include <errno.h>
   #include <stdlib.h>
}

namespace corelinux
{
   // Retreive the real user id for the current process
   
   UserIdentifier Environment::getUserId( void ) 
   {
      return UserIdentifier(getuid());
   }
   
   // Retreive the effective user id for the current process
   
   UserIdentifier Environment::getEffectiveUserId( void )
   {
      return UserIdentifier(geteuid());
   }
   
   // Retreive the real group id for the current process
   
   GroupIdentifier Environment::getGroupId( void )
   {
      return GroupIdentifier(getgid());
   }
   
   // Retreive the effective group id for the current process
   
   GroupIdentifier Environment::getEffectiveGroupId( void )
   {
      return GroupIdentifier(getegid());
   }

   // Retrieve the process group id for the current process

   ProcessIdentifier Environment::getProcessGroupId( void )
   {
      return ProcessIdentifier(getpgrp());
   }

   // Retrieve the process group id for a specific process

   ProcessIdentifier Environment::getProcessGroupId( ProcessIdentifierRef aPid )
   {
      return ProcessIdentifier(__getpgid((Int)aPid));
   }

   // Get variable from environment

   CharPtr  Environment::getEnvironmentValue( CharCptr aName )
   {
      return getenv(aName);
   }

   // Put name=value into the current environment

   Int   Environment::setEnvironmentNameValue( CharPtr aNameValue )
   {
      return putenv( aNameValue );
   }

   //
   // Routine for creating common file
   //

   Int   Environment::setupCommonAccess
      ( 
         CharCptr aName, 
         const CreateDisposition &aMode 
      )
   {
      REQUIRE( aName != NULLPTR );

      struct   stat     statRet;
               Int      results(0);
               int      errRet(0);

      int   stret = stat(aName,&statRet);

      //
      // File may not exist
      //

      if( stret == -1 )
      {
         errRet = Int(*__errno_location());

         //
         // If file not found and fail if not
         // exists, problem, else we try to
         // create, and result accordingly
         //

         if( errRet == ENOENT )
         {
            if( aMode == FAIL_IF_NOTEXISTS )
            {
               results = -1;
            }
            else
            {
               results = 1;

               Int fhndl = open
               ( 
                  aName, 
                  O_CREAT | O_EXCL, 
                  OWNER_ALL | GROUP_ALL | PUBLIC_ALL 
               );

               if( fhndl != -1 )
               {
                  close(fhndl);
               }
               else
               {
                  results = fhndl;
               }
            }
         }

         // Any other error is a failure to create

         else
         {
            results = -1;
         }
      }

      // If the file was found

      else
      {
         if( aMode == FAIL_IF_EXISTS )
         {
            results = -1;
         }
         else
         {
            ;  // do nothing
         }
      }

      return results;
   }

   //
   // Routine for removing semaphore file
   //

   Int  Environment::removeCommonAccess( CharCptr aName )
   {
      REQUIRE( aName != NULLPTR );

      struct   stat     statRet;
               Int      stret( stat(aName,&statRet) );

      if( stret != -1 )
      {
         stret = unlink(aName);
      }
      else
      {
         ;  // do nothing, bad check
      }

      return stret;
   }

   void Environment::setThreadPriority ( ProcessIdentifier pid, Int prio )
   {
      setpriority (PRIO_PROCESS, pid, prio);
   }

   Int Environment::getThreadPriority ( ProcessIdentifier pid )
   {
      return getpriority (PRIO_PROCESS, pid);
   }
}

/*
   Common rcs information do not modify
   $Author: dulimart $
   $Revision: 1.4 $
   $Date: 2001/03/20 04:09:11 $
   $Locker:  $
*/

